// $Id: CPresetIO.cpp,v 1.5 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CPresetIO.hpp"
using Exponent::IO::CPresetIO;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CPresetIO, CCountedObject);

//  ===========================================================================
CPresetIO::CPresetIO() : m_isOpenForRead(false), m_isOpenForWrite(false)
{
	EXPONENT_CLASS_CONSTRUCTION(CPresetIO);
	m_isOpenForRead  = false;
	m_isOpenForWrite = false;
}

//  ===========================================================================
CPresetIO::~CPresetIO()
{
	EXPONENT_CLASS_DESTRUCTION(CPresetIO);
	m_stream.closeStream();
}

//  ===========================================================================
bool CPresetIO::open(const CTextStream::EStreamMode streamMode, const CString &pluginName, const CSystemString &presetPath)
{
	// Open the file
	bool opened = m_stream.openStream(presetPath, streamMode);

	// check it opened properly
	if (!opened)
	{
		return false;
	}

	// depending on the stream mode we either  read or write the plugin name
	if (streamMode == CTextStream::e_input)
	{
		m_isOpenForRead  = true;
		m_isOpenForWrite = false;

		// Read the opening tag
		return this->readOpenTag(pluginName);
	}
	else if (streamMode == CTextStream::e_output)
	{
		m_isOpenForRead  = false;
		m_isOpenForWrite = true;

		return this->writeOpenTag(pluginName);
	}

	// We have failed somewhere along the way
	return false;
}

//  ===========================================================================
bool CPresetIO::openHeadChunk()
{
	if (m_isOpenForRead)
	{
		return this->readOpenTag("head");
	}
	else if (m_isOpenForWrite)
	{
		return this->writeOpenTag("head");
	}
	return false;
}

//  ===========================================================================
bool CPresetIO::openBodyChunk()
{
	if (m_isOpenForRead)
	{
		return this->readOpenTag("body");
	}
	else if (m_isOpenForWrite)
	{
		return this->writeOpenTag("body");
	}
	return false;
}

//  ===========================================================================
bool CPresetIO::closeHeadChunk()
{
	if (m_isOpenForRead)
	{
		return this->readCloseTag("head");
	}
	else if (m_isOpenForWrite)
	{
		return this->writeCloseTag("head");
	}
	return false;
}

//  ===========================================================================
bool CPresetIO::closeBodyChunk()
{
	if (m_isOpenForRead)
	{
		return this->readCloseTag("body");
	}
	else if (m_isOpenForWrite)
	{
		return this->writeCloseTag("body");
	}
	return false;
}

//  ===========================================================================
bool CPresetIO::close(const CString &pluginName)
{
	bool valid = false;
	if (m_isOpenForRead)
	{
		valid = this->readCloseTag(pluginName);
	}
	else if (m_isOpenForWrite)
	{
		valid = this->writeCloseTag(pluginName);
	}
	else
	{
		valid = true;
	}

	// No longer open
	m_isOpenForWrite = false;
	m_isOpenForRead  = false;

	// Close the stream
	m_stream.closeStream();

	// Return the validity
	return valid;
}

//  ===========================================================================
void CPresetIO::close()
{
	// Close the stream
	m_stream.closeStream();
}

//  ===========================================================================
bool CPresetIO::readHeadChunk(TStringCountedPointerArray &array)
{
	return this->readChunkToStrings("head", array);
}

//  ===========================================================================
bool CPresetIO::writeVersionNumber(const long number)
{
	// Check the stream is open
	if (!m_stream.isStreamOpen())
	{
		return false;
	}

	// Write the opening tag
	if (!this->writeOpenTag("version"))
	{
		return false;
	}

	// Write the string
	m_stream << number << "\n";

	// Write the closing tag
	if (!this->writeCloseTag("version"))
	{
		return false;
	}

	return true;
}

//  ===========================================================================
bool CPresetIO::readVersionNumber(const long expectedVersion)
{
	// Read the opening tag
	if (!this->readOpenTag("version"))
	{
		return false;
	}

	// The version on disk
	long version;

	// Stream the version in
	m_stream >> version;

	// Read the closing tag
	if (!this->readCloseTag("version"))
	{
		return false;
	}

	// Check if the stream version is the same
	return (version == expectedVersion);
}

//  ===========================================================================
bool CPresetIO::readVersionNumber(const long *expectedVersion, const long numberOfVersions, long &actualVersion)
{
	// Read the opening tag
	if (!this->readOpenTag("version"))
	{
		return false;
	}

	// The version on disk
	long version;

	// Stream the version in
	m_stream >> version;

	// Read the closing tag
	if (!this->readCloseTag("version"))
	{
		return false;
	}

	// Check if the stream version is the same
	for (long i = 0; i < numberOfVersions; i++)
	{
		if (version == expectedVersion[i])
		{
			actualVersion = version;
			return true;
		}
	}

	return false;
}

//  ===========================================================================
bool CPresetIO::readParameterChunk(const long numberOfParameters, double *parameters)
{
	return this->readChunkToDoubles("parameters", numberOfParameters, parameters);
}

//  ===========================================================================
bool CPresetIO::readChunkToDoubles(const CString &tag, const long sizeOfChunk, double *buffer)
{
	// Read the opening tag
	if (!this->readOpenTag(tag))
	{
		return false;
	}

	// Close tag
	CString closeTag;
	this->constructCloseTag(closeTag, tag);

	// The string we are going to read in
	CString theString;

	// are we valid
	bool finished = false;
	long index    = 0;

	// Now loop and read everything in (<= so that we get the closing tag)
	while (!finished && index <= sizeOfChunk)
	{
		// Stream in
		m_stream >> theString;

		// If the stirng is not the closing tag
		if (theString != closeTag)
		{
			// Then we want to convert it
			buffer[index++] = atof(theString.getString());
		}
		else
		{
			finished = true;
		}
	}

	// Check if we complete properly
	return finished;
}

//  ===========================================================================
bool CPresetIO::readChunkToStrings(const CString &tag, TStringCountedPointerArray &array)
{
	// Check its valid
	if (!this->readOpenTag(tag))
	{
		return false;
	}

	// Close tag
	CString closeTag;
	this->constructCloseTag(closeTag, tag);

	// The string we are going to read in
	CString theString;

	// are we valid
	bool finished = false;

	// Now loop and read everything in
	while (!finished)
	{
		// Stream in
		m_stream >> theString;

		// If the stirng is not the closing tag
		if (theString != closeTag)
		{
			// Then we want to convert it
			array.addElement(new CString(theString));
		}
		else
		{
			finished = true;
		}
	}

	// Check if we complete properly
	return finished;
}

//  ===========================================================================
bool CPresetIO::readChunkToLongs(const CString &tag, const long sizeOfChunk, long *buffer)
{
	// Read the opening tag
	if (!this->readOpenTag(tag))
	{
		return false;
	}

	// Close tag
	CString closeTag;
	this->constructCloseTag(closeTag, tag);

	// The string we are going to read in
	CString theString;

	// are we valid
	bool finished = false;
	long index    = 0;

	// Now loop and read everything in (<= so that we get the closing tag)
	while (!finished && index <= sizeOfChunk)
	{
		// Stream in
		m_stream >> theString;

		// If the stirng is not the closing tag
		if (theString != closeTag)
		{
			// Then we want to convert it
			buffer[index++] = atol(theString.getString());
		}
		else
		{
			finished = true;
		}
	}

	// Check if we complete properly
	return finished;
}


//  ===========================================================================
bool CPresetIO::writeHeadChunk(TStringCountedPointerArray &array)
{
	return this->writeChunkToStrings("head", array);
}

//  ===========================================================================
bool CPresetIO::writeParameterChunk(const long numberOfParameters, const double *parameters)
{
	return this->writeChunkToDoubles("parameters", numberOfParameters, parameters);
}

//  ===========================================================================
bool CPresetIO::writeChunkToDoubles(const CString &tag, const long sizeOfChunk, const double *buffer)
{
	// Check the stream is open
	if (!m_stream.isStreamOpen())
	{
		return false;
	}

	// Write the opening tag
	if (!this->writeOpenTag(tag))
	{
		return false;
	}

	// Now output the chunk
	for (long i = 0; i < sizeOfChunk; i++)
	{
		m_stream << buffer[i] << "\n";
	}

	// Write the closing tag
	if (!this->writeCloseTag(tag))
	{
		return false;
	}

	// We have complete sucessfully!
	return true;
}

//  ===========================================================================
bool CPresetIO::writeChunkToStrings(const CString &tag, TStringCountedPointerArray &array)
{
	// Check the stream is open
	if (!m_isOpenForWrite)
	{
		return false;
	}

	// Write the opening tag
	if (!this->writeOpenTag(tag))
	{
		return false;
	}

	// Now output the chunk
	for (long i = 0; i < array.getArraySize(); i++)
	{
		// GEt the string
		CString *string = array.elementAtIndex(i);

		// Check its valid
		if (string)
		{
			// Output
			m_stream << *string << "\n";
		}
	}

	// Write the closing tag
	if (!this->writeCloseTag(tag))
	{
		return false;
	}

	// We have complete sucessfully!
	return true;
}

//  ===========================================================================
bool CPresetIO::writeChunkToLongs(const CString &tag, const long sizeOfChunk, long *buffer)
{
	// Check the stream is open
	if (!m_stream.isStreamOpen())
	{
		return false;
	}

	// Write the opening tag
	if (!this->writeOpenTag(tag))
	{
		return false;
	}

	// Now output the chunk
	for (long i = 0; i < sizeOfChunk; i++)
	{
		m_stream << buffer[i] << "\n";
	}

	// Write the closing tag
	if (!this->writeCloseTag(tag))
	{
		return false;
	}

	// We have complete sucessfully!
	return true;
}

//  ===========================================================================
bool CPresetIO::writeOpenTag(const CString &tag)
{
	// Check the stream is open
	if (!m_isOpenForWrite)
	{
		return false;
	}

	// Create the opening tag
	CString openTag;
	this->constructOpenTag(openTag, tag);

	// Stream it out
	m_stream << openTag << "\n";

	// We're dont
	return true;
}

//  ===========================================================================
bool CPresetIO::writeCloseTag(const CString &tag)
{
	// Check the stream is open
	if (!m_isOpenForWrite)
	{
		return false;
	}

	// Close tag
	CString closeTag;
	this->constructCloseTag(closeTag, tag);

	// Stream it out
	m_stream << closeTag << "\n";

	// We're dont
	return true;
}

//  ===========================================================================
bool CPresetIO::readOpenTag(const CString &tag)
{
	// Check the stream is open
	if (!m_isOpenForRead)
	{
		return false;
	}

	// Create the opening tag
	CString openTag;
	this->constructOpenTag(openTag, tag);

	// The string we are going to read
	CString theString;

	// Stream in
	m_stream >> theString;

	// Check if the tag matches
	return (theString == openTag);
}

//  ===========================================================================
bool CPresetIO::readCloseTag(const CString &tag)
{
	// Check the stream is open
	if (!m_isOpenForRead)
	{
		return false;
	}

	// Create the closing tag
	CString closeTag;
	this->constructCloseTag(closeTag, tag);

	// The string we are going to read
	CString theString;

	// Stream in
	m_stream >> theString;

	// Check if the tag matches
	return (theString == closeTag);
}

//  ===========================================================================
void CPresetIO::constructOpenTag(CString &openTag, const CString &tag)
{
	openTag = "<";
	openTag.appendString(tag);
	openTag.appendString(">");
}

//  ===========================================================================
void CPresetIO::constructCloseTag(CString &closeTag, const CString &tag)
{
	closeTag = "</";
	closeTag.appendString(tag);
	closeTag.appendString(">");
}
